"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const issue_1 = require("../../../constants/issue");
const media_1 = require("../../../constants/media");
const datasource_1 = require("../../../datasource");
const MediaRequest_1 = __importDefault(require("../../../entity/MediaRequest"));
const User_1 = require("../../../entity/User");
const UserPushSubscription_1 = require("../../../entity/UserPushSubscription");
const settings_1 = require("../../../lib/settings");
const logger_1 = __importDefault(require("../../../logger"));
const web_push_1 = __importDefault(require("web-push"));
const __1 = require("..");
const agent_1 = require("./agent");
class WebPushAgent extends agent_1.BaseAgent {
    getSettings() {
        if (this.settings) {
            return this.settings;
        }
        const settings = (0, settings_1.getSettings)();
        return settings.notifications.agents.webpush;
    }
    getNotificationPayload(type, payload) {
        const mediaType = payload.media
            ? payload.media.mediaType === media_1.MediaType.MOVIE
                ? 'movie'
                : 'series'
            : undefined;
        const is4k = payload.request?.is4k;
        const issueType = payload.issue
            ? payload.issue.issueType !== issue_1.IssueType.OTHER
                ? `${issue_1.IssueTypeName[payload.issue.issueType].toLowerCase()} issue`
                : 'issue'
            : undefined;
        let message;
        switch (type) {
            case __1.Notification.TEST_NOTIFICATION:
                message = payload.message;
                break;
            case __1.Notification.MEDIA_AUTO_REQUESTED:
                message = `Automatically submitted a new ${is4k ? '4K ' : ''}${mediaType} request.`;
                break;
            case __1.Notification.MEDIA_APPROVED:
                message = `Your ${is4k ? '4K ' : ''}${mediaType} request has been approved.`;
                break;
            case __1.Notification.MEDIA_AUTO_APPROVED:
                message = `Automatically approved a new ${is4k ? '4K ' : ''}${mediaType} request from ${payload.request?.requestedBy.displayName}.`;
                break;
            case __1.Notification.MEDIA_AVAILABLE:
                message = `Your ${is4k ? '4K ' : ''}${mediaType} request is now available!`;
                break;
            case __1.Notification.MEDIA_DECLINED:
                message = `Your ${is4k ? '4K ' : ''}${mediaType} request was declined.`;
                break;
            case __1.Notification.MEDIA_FAILED:
                message = `Failed to process ${is4k ? '4K ' : ''}${mediaType} request.`;
                break;
            case __1.Notification.MEDIA_PENDING:
                message = `Approval required for a new ${is4k ? '4K ' : ''}${mediaType} request from ${payload.request?.requestedBy.displayName}.`;
                break;
            case __1.Notification.ISSUE_CREATED:
                message = `A new ${issueType} was reported by ${payload.issue?.createdBy.displayName}.`;
                break;
            case __1.Notification.ISSUE_COMMENT:
                message = `${payload.comment?.user.displayName} commented on the ${issueType}.`;
                break;
            case __1.Notification.ISSUE_RESOLVED:
                message = `The ${issueType} was marked as resolved by ${payload.issue?.modifiedBy?.displayName}!`;
                break;
            case __1.Notification.ISSUE_REOPENED:
                message = `The ${issueType} was reopened by ${payload.issue?.modifiedBy?.displayName}.`;
                break;
            default:
                return {
                    notificationType: __1.Notification[type],
                    subject: 'Unknown',
                };
        }
        const actionUrl = payload.issue
            ? `/issues/${payload.issue.id}`
            : payload.media
                ? `/${payload.media.mediaType}/${payload.media.tmdbId}`
                : undefined;
        const actionUrlTitle = actionUrl
            ? `View ${payload.issue ? 'Issue' : 'Media'}`
            : undefined;
        return {
            notificationType: __1.Notification[type],
            subject: payload.subject,
            message,
            image: payload.image,
            requestId: payload.request?.id,
            actionUrl,
            actionUrlTitle,
            pendingRequestsCount: payload.pendingRequestsCount,
            isAdmin: payload.isAdmin,
        };
    }
    shouldSend() {
        if (this.getSettings().enabled) {
            return true;
        }
        return false;
    }
    async send(type, payload) {
        const userRepository = (0, datasource_1.getRepository)(User_1.User);
        const userPushSubRepository = (0, datasource_1.getRepository)(UserPushSubscription_1.UserPushSubscription);
        const settings = (0, settings_1.getSettings)();
        const pushSubs = [];
        const mainUser = await userRepository.findOne({ where: { id: 1 } });
        const requestRepository = (0, datasource_1.getRepository)(MediaRequest_1.default);
        const pendingRequests = await requestRepository.find({
            where: { status: media_1.MediaRequestStatus.PENDING },
        });
        const webPushNotification = async (pushSub, notificationPayload) => {
            logger_1.default.debug('Sending web push notification', {
                label: 'Notifications',
                recipient: pushSub.user.displayName,
                type: __1.Notification[type],
                subject: payload.subject,
            });
            try {
                await web_push_1.default.sendNotification({
                    endpoint: pushSub.endpoint,
                    keys: {
                        auth: pushSub.auth,
                        p256dh: pushSub.p256dh,
                    },
                }, notificationPayload);
            }
            catch (e) {
                logger_1.default.error('Error sending web push notification; removing subscription', {
                    label: 'Notifications',
                    recipient: pushSub.user.displayName,
                    type: __1.Notification[type],
                    subject: payload.subject,
                    errorMessage: e.message,
                });
                // Failed to send notification so we need to remove the subscription
                userPushSubRepository.remove(pushSub);
            }
        };
        if (payload.notifyUser &&
            // Check if user has webpush notifications enabled and fallback to true if undefined
            // since web push should default to true
            (payload.notifyUser.settings?.hasNotificationType(settings_1.NotificationAgentKey.WEBPUSH, type) ??
                true)) {
            const notifySubs = await userPushSubRepository.find({
                where: { user: { id: payload.notifyUser.id } },
            });
            pushSubs.push(...notifySubs);
        }
        if (payload.notifyAdmin ||
            type === __1.Notification.MEDIA_APPROVED ||
            type === __1.Notification.MEDIA_DECLINED) {
            const users = await userRepository.find();
            const manageUsers = users.filter((user) => 
            // Check if user has webpush notifications enabled and fallback to true if undefined
            // since web push should default to true
            (user.settings?.hasNotificationType(settings_1.NotificationAgentKey.WEBPUSH, type) ??
                true) &&
                (0, __1.shouldSendAdminNotification)(type, user, payload));
            const allSubs = await userPushSubRepository
                .createQueryBuilder('pushSub')
                .leftJoinAndSelect('pushSub.user', 'user')
                .where('pushSub.userId IN (:...users)', {
                users: manageUsers.map((user) => user.id),
            })
                .getMany();
            // We only want to send the custom notification when type is approved or declined
            // Otherwise, default to the normal notification
            if (type === __1.Notification.MEDIA_APPROVED ||
                type === __1.Notification.MEDIA_DECLINED) {
                if (mainUser && allSubs.length > 0) {
                    web_push_1.default.setVapidDetails(`mailto:${mainUser.email}`, settings.vapidPublic, settings.vapidPrivate);
                    // Custom payload only for updating the app badge
                    const notificationBadgePayload = Buffer.from(JSON.stringify(this.getNotificationPayload(type, {
                        subject: payload.subject,
                        notifySystem: false,
                        notifyAdmin: true,
                        isAdmin: true,
                        pendingRequestsCount: pendingRequests.length,
                    })), 'utf-8');
                    await Promise.all(allSubs.map(async (sub) => {
                        webPushNotification(sub, notificationBadgePayload);
                    }));
                }
            }
            else {
                pushSubs.push(...allSubs);
            }
        }
        if (mainUser && pushSubs.length > 0) {
            web_push_1.default.setVapidDetails(`mailto:${mainUser.email}`, settings.vapidPublic, settings.vapidPrivate);
            if (type === __1.Notification.MEDIA_PENDING) {
                payload = { ...payload, pendingRequestsCount: pendingRequests.length };
            }
            const notificationPayload = Buffer.from(JSON.stringify(this.getNotificationPayload(type, payload)), 'utf-8');
            await Promise.all(pushSubs.map(async (sub) => {
                webPushNotification(sub, notificationPayload);
            }));
        }
        return true;
    }
}
exports.default = WebPushAgent;
