"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.blacklistAdd = void 0;
const media_1 = require("../constants/media");
const datasource_1 = require("../datasource");
const Blacklist_1 = require("../entity/Blacklist");
const Media_1 = __importDefault(require("../entity/Media"));
const permissions_1 = require("../lib/permissions");
const logger_1 = __importDefault(require("../logger"));
const auth_1 = require("../middleware/auth");
const express_1 = require("express");
const typeorm_1 = require("typeorm");
const zod_1 = require("zod");
const blacklistRoutes = (0, express_1.Router)();
exports.blacklistAdd = zod_1.z.object({
    tmdbId: zod_1.z.coerce.number(),
    mediaType: zod_1.z.nativeEnum(media_1.MediaType),
    title: zod_1.z.coerce.string().optional(),
    user: zod_1.z.coerce.number(),
});
const blacklistGet = zod_1.z.object({
    take: zod_1.z.coerce.number().int().positive().default(25),
    skip: zod_1.z.coerce.number().int().nonnegative().default(0),
    search: zod_1.z.string().optional(),
    filter: zod_1.z.enum(['all', 'manual', 'blacklistedTags']).optional(),
});
blacklistRoutes.get('/', (0, auth_1.isAuthenticated)([permissions_1.Permission.MANAGE_BLACKLIST, permissions_1.Permission.VIEW_BLACKLIST], {
    type: 'or',
}), async (req, res, next) => {
    const { take, skip, search, filter } = blacklistGet.parse(req.query);
    try {
        let query = (0, datasource_1.getRepository)(Blacklist_1.Blacklist)
            .createQueryBuilder('blacklist')
            .leftJoinAndSelect('blacklist.user', 'user')
            .where('1 = 1'); // Allow use of andWhere later
        switch (filter) {
            case 'manual':
                query = query.andWhere('blacklist.blacklistedTags IS NULL');
                break;
            case 'blacklistedTags':
                query = query.andWhere('blacklist.blacklistedTags IS NOT NULL');
                break;
        }
        if (search) {
            query = query.andWhere('blacklist.title like :title', {
                title: `%${search}%`,
            });
        }
        const [blacklistedItems, itemsCount] = await query
            .orderBy('blacklist.createdAt', 'DESC')
            .take(take)
            .skip(skip)
            .getManyAndCount();
        return res.status(200).json({
            pageInfo: {
                pages: Math.ceil(itemsCount / take),
                pageSize: take,
                results: itemsCount,
                page: Math.ceil(skip / take) + 1,
            },
            results: blacklistedItems,
        });
    }
    catch (error) {
        logger_1.default.error('Something went wrong while retrieving blacklisted items', {
            label: 'Blacklist',
            errorMessage: error.message,
        });
        return next({
            status: 500,
            message: 'Unable to retrieve blacklisted items.',
        });
    }
});
blacklistRoutes.get('/:id', (0, auth_1.isAuthenticated)([permissions_1.Permission.MANAGE_BLACKLIST], {
    type: 'or',
}), async (req, res, next) => {
    try {
        const blacklisteRepository = (0, datasource_1.getRepository)(Blacklist_1.Blacklist);
        const blacklistItem = await blacklisteRepository.findOneOrFail({
            where: { tmdbId: Number(req.params.id) },
        });
        return res.status(200).send(blacklistItem);
    }
    catch (e) {
        if (e instanceof typeorm_1.EntityNotFoundError) {
            return next({
                status: 401,
                message: e.message,
            });
        }
        return next({ status: 500, message: e.message });
    }
});
blacklistRoutes.post('/', (0, auth_1.isAuthenticated)([permissions_1.Permission.MANAGE_BLACKLIST], {
    type: 'or',
}), async (req, res, next) => {
    try {
        const values = exports.blacklistAdd.parse(req.body);
        await Blacklist_1.Blacklist.addToBlacklist({
            blacklistRequest: values,
        });
        return res.status(201).send();
    }
    catch (error) {
        if (!(error instanceof Error)) {
            return;
        }
        if (error instanceof typeorm_1.QueryFailedError) {
            switch (error.driverError.errno) {
                case 19:
                    return next({ status: 412, message: 'Item already blacklisted' });
                default:
                    logger_1.default.warn('Something wrong with data blacklist', {
                        tmdbId: req.body.tmdbId,
                        mediaType: req.body.mediaType,
                        label: 'Blacklist',
                    });
                    return next({ status: 409, message: 'Something wrong' });
            }
        }
        return next({ status: 500, message: error.message });
    }
});
blacklistRoutes.delete('/:id', (0, auth_1.isAuthenticated)([permissions_1.Permission.MANAGE_BLACKLIST], {
    type: 'or',
}), async (req, res, next) => {
    try {
        const blacklisteRepository = (0, datasource_1.getRepository)(Blacklist_1.Blacklist);
        const blacklistItem = await blacklisteRepository.findOneOrFail({
            where: { tmdbId: Number(req.params.id) },
        });
        await blacklisteRepository.remove(blacklistItem);
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        const mediaItem = await mediaRepository.findOneOrFail({
            where: { tmdbId: Number(req.params.id) },
        });
        await mediaRepository.remove(mediaItem);
        return res.status(204).send();
    }
    catch (e) {
        if (e instanceof typeorm_1.EntityNotFoundError) {
            return next({
                status: 401,
                message: e.message,
            });
        }
        return next({ status: 500, message: e.message });
    }
});
exports.default = blacklistRoutes;
